import { useTranslations } from "next-intl";

/**
 * Custom hook to get translated menu items
 * Handles both title and description translations
 */
export function useMenuTranslations() {
  const t = useTranslations("menu");

  /**
   * Navigate nested object to get value
   */
  const getNestedValue = (obj: any, path: string): any => {
    const keys = path.split('.');
    let current = obj;
    for (const key of keys) {
      if (current && typeof current === 'object' && key in current) {
        current = current[key];
      } else {
        return undefined;
      }
    }
    return current;
  };

  /**
   * Get translated title for a menu item
   */
  const getTitle = (item: any): string => {
    // Return fallback immediately if no valid key
    if (!item?.key || typeof item.key !== 'string' || item.key.trim() === '') {
      return item?.title || "";
    }

    // Validate key format before attempting translation
    const cleanKey = item.key.trim();
    if (cleanKey.startsWith('.') || cleanKey.endsWith('.') || cleanKey.includes('..')) {
      return item?.title || "";
    }

    try {
      // Convert key from "admin-affiliate-system" to nested path parts
      const pathParts = cleanKey.replace(/-/g, '.').split('.').filter(Boolean);

      // Must have at least one valid path part
      if (pathParts.length === 0) {
        return item?.title || "";
      }

      const translationPath = [...pathParts, 'title'].join('.');

      // Get the raw menu translations object
      const menuTranslations = t.raw('.');

      // Navigate the nested structure
      const translated = getNestedValue(menuTranslations, translationPath);

      return translated || item.title || "";
    } catch (error) {
      // Silently return fallback on any error
      return item.title || "";
    }
  };

  /**
   * Get translated description for a menu item
   */
  const getDescription = (item: any): string => {
    // Return fallback immediately if no valid key or description
    if (!item?.key || typeof item.key !== 'string' || item.key.trim() === '' || !item?.description) {
      return item?.description || "";
    }

    // Validate key format before attempting translation
    const cleanKey = item.key.trim();
    if (cleanKey.startsWith('.') || cleanKey.endsWith('.') || cleanKey.includes('..')) {
      return item?.description || "";
    }

    try {
      // Convert key from "admin-affiliate-system" to nested path parts
      const pathParts = cleanKey.replace(/-/g, '.').split('.').filter(Boolean);

      // Must have at least one valid path part
      if (pathParts.length === 0) {
        return item?.description || "";
      }

      const translationPath = [...pathParts, 'description'].join('.');

      // Get the raw menu translations object
      const menuTranslations = t.raw('.');

      // Navigate the nested structure
      const translated = getNestedValue(menuTranslations, translationPath);

      return translated || item.description || "";
    } catch (error) {
      // Silently return fallback on any error
      return item.description || "";
    }
  };

  return { getTitle, getDescription };
}
