import { ethers } from "ethers";
import { ERC721_NFT_ABI } from "./nft-abis";

export interface MintTransactionParams {
  contractAddress: string;
  recipientAddress: string;
  tokenURI: string; // Metadata URI (IPFS or image URL)
  mintPrice: string; // in ETH
  chain: string;
}

export interface MintTransactionResult {
  success: boolean;
  transactionHash?: string;
  tokenId?: string;
  blockNumber?: number;
  gasUsed?: string;
  error?: string;
}

/**
 * Prepare and send NFT mint transaction via user's Web3 wallet
 * Supports both MetaMask and WalletConnect via Reown AppKit
 * This implements true Web3 flow where users sign their own transactions
 */
export async function mintNFTViaWeb3(
  params: MintTransactionParams
): Promise<MintTransactionResult> {
  try {
    const { contractAddress, recipientAddress, tokenURI, mintPrice, chain } = params;

    // Get provider from Wagmi config
    const { config } = await import("@/config/wallet");
    const { getConnectorClient } = await import("wagmi/actions");
    const { BrowserProvider } = await import("ethers");

    // Get the connected wallet client from Wagmi
    const connectorClient = await getConnectorClient(config);

    if (!connectorClient || !connectorClient.account) {
      throw new Error("No wallet provider found. Please connect your wallet.");
    }

    const userAddress = connectorClient.account.address;

    // Verify the recipient address matches the connected wallet
    if (recipientAddress.toLowerCase() !== userAddress.toLowerCase()) {
      throw new Error("Recipient address must match connected wallet address.");
    }

    // Create ethers provider from Wagmi connector
    const provider = new BrowserProvider(connectorClient.transport);
    const signer = await provider.getSigner();

    // Create contract instance
    const nftContract = new ethers.Contract(
      contractAddress,
      ERC721_NFT_ABI,
      signer
    );

    // Prepare transaction options
    const txOptions: any = {};

    // Add mint price if required
    if (mintPrice && parseFloat(mintPrice) > 0) {
      txOptions.value = ethers.parseEther(mintPrice);
    }

    console.log("[NFT MINT] Preparing transaction:", {
      contractAddress,
      tokenURI,
      mintPrice,
      chain,
    });

    // Estimate gas
    try {
      const gasEstimate = await nftContract.mint.estimateGas(
        tokenURI,
        txOptions
      );
      // Add 20% buffer to gas estimate
      txOptions.gasLimit = (gasEstimate * BigInt(120)) / BigInt(100);
      console.log("[NFT MINT] Gas estimate:", gasEstimate.toString());
    } catch (gasError) {
      console.error("[NFT MINT] Gas estimation failed:", gasError);
      // Continue without gas limit - MetaMask will estimate
    }

    // Send transaction - this will prompt MetaMask for user signature
    console.log("[NFT MINT] Requesting user signature...");
    const tx = await nftContract.mint(tokenURI, txOptions);

    console.log("[NFT MINT] Transaction sent:", tx.hash);
    console.log("[NFT MINT] Waiting for confirmation...");

    // Wait for transaction confirmation
    const receipt = await tx.wait();

    if (!receipt) {
      throw new Error("Transaction receipt not found");
    }

    console.log("[NFT MINT] Transaction confirmed in block:", receipt.blockNumber);

    // Extract token ID from Transfer event logs
    let tokenId: string | undefined;
    const transferEventTopic = ethers.id("Transfer(address,address,uint256)");

    for (const log of receipt.logs) {
      if (log.topics[0] === transferEventTopic) {
        // Token ID is the third topic in Transfer event
        tokenId = BigInt(log.topics[3]).toString();
        console.log("[NFT MINT] Token ID:", tokenId);
        break;
      }
    }

    // Calculate gas cost
    const gasUsed = receipt.gasUsed;
    const gasPrice = tx.gasPrice || receipt.gasPrice || BigInt(0);
    const gasCost = gasUsed * gasPrice;
    const gasCostEth = ethers.formatEther(gasCost);

    console.log("[NFT MINT] Gas used:", gasUsed.toString());
    console.log("[NFT MINT] Gas cost:", gasCostEth, "ETH");

    return {
      success: true,
      transactionHash: receipt.hash,
      tokenId,
      blockNumber: receipt.blockNumber,
      gasUsed: gasUsed.toString(),
    };
  } catch (error: any) {
    console.error("[NFT MINT] Error:", error);

    // Handle user rejection
    if (error.code === 4001 || error.code === "ACTION_REJECTED") {
      return {
        success: false,
        error: "Transaction was rejected by user",
      };
    }

    // Handle insufficient funds
    if (
      error.code === "INSUFFICIENT_FUNDS" ||
      error.message?.includes("insufficient funds")
    ) {
      return {
        success: false,
        error: "Insufficient funds to complete the transaction",
      };
    }

    // Handle gas estimation errors
    if (
      error.code === "UNPREDICTABLE_GAS_LIMIT" ||
      error.message?.includes("gas")
    ) {
      return {
        success: false,
        error: "Gas estimation failed. Please check contract state or try again.",
      };
    }

    return {
      success: false,
      error: error.message || "Failed to mint NFT",
    };
  }
}

/**
 * Get current gas price for the connected network
 */
export async function getCurrentGasPrice(): Promise<string | null> {
  try {
    // Get provider from wallet store
    const { useWalletStore } = await import("@/store/nft/wallet-store");
    const walletStore = useWalletStore.getState();
    const walletProvider = await walletStore.getProvider();

    if (!walletProvider) {
      return null;
    }

    const provider = new ethers.BrowserProvider(walletProvider);
    const feeData = await provider.getFeeData();

    if (feeData.gasPrice) {
      return ethers.formatUnits(feeData.gasPrice, "gwei");
    }

    return null;
  } catch (error) {
    console.error("[GAS PRICE] Error fetching gas price:", error);
    return null;
  }
}

/**
 * Estimate gas cost for minting an NFT
 */
export async function estimateMintGasCost(
  contractAddress: string,
  recipientAddress: string,
  mintPrice: string
): Promise<{
  gasLimit: string;
  gasPrice: string;
  totalCost: string;
  totalCostEth: string;
} | null> {
  try {
    // Get provider from wallet store
    const { useWalletStore } = await import("@/store/nft/wallet-store");
    const walletStore = useWalletStore.getState();
    const walletProvider = await walletStore.getProvider();

    if (!walletProvider) {
      return null;
    }

    const provider = new ethers.BrowserProvider(walletProvider);
    const signer = await provider.getSigner();

    const nftContract = new ethers.Contract(
      contractAddress,
      ERC721_NFT_ABI,
      signer
    );

    const txOptions: any = {};
    if (mintPrice && parseFloat(mintPrice) > 0) {
      txOptions.value = ethers.parseEther(mintPrice);
    }

    // Estimate gas
    const gasEstimate = await nftContract.mint.estimateGas(
      recipientAddress,
      txOptions
    );

    // Get current gas price
    const feeData = await provider.getFeeData();
    const gasPrice = feeData.gasPrice || BigInt(0);

    // Calculate total cost (gas + mint price)
    const gasCost = gasEstimate * gasPrice;
    const mintCost = txOptions.value || BigInt(0);
    const totalCost = gasCost + mintCost;

    return {
      gasLimit: gasEstimate.toString(),
      gasPrice: ethers.formatUnits(gasPrice, "gwei"),
      totalCost: totalCost.toString(),
      totalCostEth: ethers.formatEther(totalCost),
    };
  } catch (error) {
    console.error("[GAS ESTIMATE] Error estimating gas:", error);
    return null;
  }
}
